<?php

/**
 * FORQY Less Compiler
 */

if ( !class_exists( 'forqy_less' ) ) {

	// Add on Init to Support Theme Customizer
	add_action( 'init', array(
		'forqy_less',
		'instance'
	) );

	/**
	 * Enables the use of LESS in WordPress
	 */
	class forqy_less {

		/**
		 * @var forqy_less|null Reusable object instance.
		 */
		protected static ?forqy_less $instance = null;

		/**
		 * Creates a new instance. Called on 'after_setup_theme'.
		 * May be used to access class methods from outside.
		 *
		 * @return forqy_less|null
		 */
		public static function instance(): ?forqy_less {
			null === self:: $instance and self:: $instance = new self;

			return self:: $instance;
		}

		/**
		 * @var array Array store of callable functions used to extend the parser
		 */
		public array $registered_functions = array();

		/**
		 * @var array Array store of function names to be removed from the compiler class
		 */
		public array $unregistered_functions = array();

		/**
		 * @var array Variables to be passed into the compiler
		 */
		public array $vars = array();

		/**
		 * @var string Compression class to use
		 */
		public string $compression = 'compressed';

		/**
		 * @var bool Whether to preserve comments when compiling
		 */
		public bool $preserve_comments = false;

		/**
		 * @var array Default import directory paths for lessc to scan
		 */
		public array $import_dirs = array();

		/**
		 * Constructor
		 */
		public function __construct() {

			// Every CSS file URL gets passed through this filter
			add_filter( 'style_loader_src', array( $this, 'parse_stylesheet' ), 100000, 2 );

			// Editor stylesheet URLs are concatenated and run through this filter
			add_filter( 'mce_css', array( $this, 'parse_editor_stylesheets' ), 100000 );

		}

		/**
		 * Less-ify the Stylesheet and Return the 'href' of the Compiled File
		 *
		 * @param $src
		 * @param $handle
		 *
		 * @return string URL of the compiled stylesheet
		 */
		public function parse_stylesheet( $src, $handle ): string {

			// Handle only .less files
			if ( !preg_match( '/\.less(\.php)?$/', preg_replace( '/\?.*$/', '', $src ) ) ) {
				return $src;
			}

			// Get file path from $src
			if ( !strstr( $src, '?' ) ) {
				$src .= '?';
			} // Prevent non-existent index warning when using list() & explode()

			// Match the URL schemes between WP_CONTENT_URL and $src,
			// so the str_replace further down will work
			$src_scheme            = parse_url( $src, PHP_URL_SCHEME );
			$wp_content_url_scheme = parse_url( WP_CONTENT_URL, PHP_URL_SCHEME );
			if ( $src_scheme != $wp_content_url_scheme ) {
				$src = set_url_scheme( $src, $wp_content_url_scheme );
			}

			list( $less_path, $query_string ) = explode( '?', str_replace( WP_CONTENT_URL, WP_CONTENT_DIR, $src ) );

			$cache = $this->get_cached_file_data( $handle );

			// Variables to pass into the compiler - default '@themeurl' variable for image urls, etc.
			$this->vars[ 'template_directory_uri' ]   = '~"' . get_template_directory_uri() . '"';
			$this->vars[ 'stylesheet_directory_uri' ] = '~"' . get_stylesheet_directory_uri() . '"';
			$this->vars[ 'less_directory_uri' ]       = '~"' . dirname( $src ) . '"';
			$this->vars                               = apply_filters( 'less_vars', $this->vars, $handle );

			// The overall "version" of the LESS file is all it's vars, src etc.
			$less_version = md5( serialize( array( $this->vars, $src ) ) );

			/**
			 * Give the ability to disable always compiling the LESS with lessc()
			 * and instead just use the $vars and $version of the LESS file to
			 * dictate whether the LESS should be (re)generated.
			 *
			 * This means we don't need to run everything through the lessc() compiler
			 * on every page load. The tradeoff is making a change in a LESS file will not
			 * necessarily cause a (re)generation, one would need to bump the $ver param
			 * on wp_enqueue_script() to cause that.
			 */
			if ( !get_option( 'forqy_less_always_compile_less', true ) ) {

				if ( ( !empty( $cache[ 'version' ] ) ) && $cache[ 'version' ] === $less_version ) {
					// restore query string it had if any
					$url = $cache[ 'url' ] . ( !empty( $query_string ) ? "?{$query_string}" : '' );
					$url = set_url_scheme( $url, $src_scheme );

					return add_query_arg( 'ver', $less_version, $url );
				}
			}

			// Automatically regenerate files if source's modified time has changed or vars have changed
			try {

				// Initialise the parser
				$less = new lessc;

				// If the cache or root path in it are invalid then regenerate
				if ( empty( $cache ) || empty( $cache[ 'less' ][ 'root' ] ) || !file_exists( $cache[ 'less' ][ 'root' ] ) ) {
					$cache = array( 'vars' => $this->vars, 'less' => $less_path );
				}

				if ( empty( $cache[ 'url' ] ) ) {
					$cache[ 'url' ] = trailingslashit( $this->get_cache_dir( false ) ) . "{$handle}.css";
				}

				// LESS config
				$less->setFormatter( apply_filters( 'forqy_less_compression', $this->compression ) );
				$less->setPreserveComments( apply_filters( 'forqy_less_preserve_comments', $this->preserve_comments ) );
				$less->setVariables( $this->vars );

				// Add directories to scan for imports
				$import_dirs = apply_filters( 'forqy_less_import_dirs', $this->import_dirs );
				if ( !empty( $import_dirs ) ) {
					foreach ( (array)$import_dirs as $dir ) {
						$less->addImportDir( $dir );
					}
				}

				// Register and unregister functions
				foreach ( $this->registered_functions as $name => $callable ) {
					$less->registerFunction( $name, $callable );
				}

				foreach ( $this->unregistered_functions as $name ) {
					$less->unregisterFunction( $name );
				}

				// Allow devs to mess around with the less object configuration
				do_action_ref_array( 'lessc', array( &$less ) );

				// $less->cachedCompile only checks for changed file modification times
				// if using the theme customizer (changed variables not files) then force compile
				if ( $this->vars !== $cache[ 'vars' ] ) {
					$force = true;
				} else {
					$force = false;
				}
				$less_cache = $less->cachedCompile( $cache[ 'less' ], apply_filters( 'forqy_less_force_compile', $force ) );

				if ( empty( $cache ) || empty( $cache[ 'less' ][ 'updated' ] ) || md5( $less_cache[ 'compiled' ] ) !== md5( $cache[ 'less' ][ 'compiled' ] ) || $this->vars !== $cache[ 'vars' ] ) {

					// Output CSS file name
					$css_path = trailingslashit( $this->get_cache_dir() ) . "{$handle}.css";

					$cache = array(
						'vars'    => $this->vars,
						'url'     => trailingslashit( $this->get_cache_dir( false ) ) . "{$handle}.css",
						'version' => $less_version,
						'less'    => null
					);

					/**
					 * If the option to not have LESS always compiled is set,
					 * then we don't store the whole less_cache in the options' table as it's
					 * not needed because we only do a comparison based off $vars and $src
					 * (which includes the $ver param).
					 *
					 * These saves space on the options' table for high performance environments.
					 */
					if ( get_option( 'forqy_less_always_compile_less', true ) ) {
						$cache[ 'less' ] = $less_cache;
					}

					$this->save_parsed_css( $css_path, $less_cache[ 'compiled' ] );
					$this->update_cached_file_data( $handle, $cache );
				}
			} catch ( Exception $ex ) {
				wp_die( $ex->getMessage() );
			}

			// Restore query string it had if any
			$url = $cache[ 'url' ] . ( !empty( $query_string ) ? "?{$query_string}" : '' );

			// Restore original url scheme
			$url = set_url_scheme( $url, $src_scheme );

			if ( get_option( 'forqy_less_always_compile_less', true ) && !empty( $less_cache ) ) {
				return add_query_arg( 'ver', $less_cache[ 'updated' ], $url );
			} else {
				return add_query_arg( 'ver', $less_version, $url );
			}

		}

		/**
		 * Get Parsed Cache Data
		 *
		 * @param $path
		 *
		 * @return mixed|null
		 */
		public function get_cached_file_data( $path ) {
			$caches = get_option( 'forqy_less_cached_files', array() );

			if ( isset( $caches[ $path ] ) ) {
				return $caches[ $path ];
			}

			return null;
		}

		/**
		 * Save Parsed CSS File
		 *
		 * @param $css_path
		 * @param $file_contents
		 */
		public function save_parsed_css( $css_path, $file_contents ) {

			if ( !apply_filters( 'forqy_less_save_css', $css_path, $file_contents ) ) {
				return;
			}

			file_put_contents( $css_path, $file_contents );
		}

		/**
		 * Update Parsed Cache Data for This File
		 *
		 * @param $path
		 * @param $file_data
		 */
		public function update_cached_file_data( $path, $file_data ) {
			$file_data[ 'less' ][ 'compiled' ] = '';

			$caches = get_option( 'forqy_less_cached_files', array() );

			$caches[ $path ] = $file_data;

			update_option( 'forqy_less_cached_files', $caches );
		}

		/**
		 * Compile Editor Stylesheets Registered via 'add_editor_style'
		 *
		 * @param $mce_css
		 *
		 * @return string
		 */
		public function parse_editor_stylesheets( $mce_css ): string {

			// Extract CSS file URLs
			$style_sheets = explode( ",", $mce_css );

			if ( count( $style_sheets ) ) {
				$compiled_css = array();

				// Loop through editor styles, any LESS files will be compiled and the compiled URL returned
				foreach ( $style_sheets as $style_sheet ) {
					$compiled_css[] = $this->parse_stylesheet( $style_sheet, $this->url_to_handle( $style_sheet ) );
				}

				$mce_css = implode( ",", $compiled_css );
			}

			// Return new URLs
			return $mce_css;
		}

		/**
		 * Get a Nice Handle to Use for the Compiled CSS File Name
		 *
		 * @param $url
		 *
		 * @return string
		 */
		public function url_to_handle( $url ): string {

			$url = parse_url( $url );
			$url = str_replace( '.less', '', basename( $url[ 'path' ] ) );
			$url = str_replace( '/', '-', $url );

			return sanitize_key( $url );
		}

		/**
		 * Get (and Create If Unavailable) the Compiled CSS Cache Directory
		 *
		 * @param bool $path
		 *
		 * @return string
		 */
		public function get_cache_dir( bool $path = true ): string {

			// Get path and url info
			$upload_dir = wp_upload_dir();
			// Get current theme
			$theme      = wp_get_theme();
			$theme_name = strtolower( esc_attr( $theme->get( 'Name' ) ) );
			$theme_name = str_replace( ' ', '-', $theme_name );
			$theme_name = preg_replace( '/-+/', '-', $theme_name );
			$theme_name = preg_replace( '/[^A-Za-z0-9\-]/', '', $theme_name );

			if ( $path ) {
				$dir = apply_filters( 'forqy_less_cache_path', path_join( $upload_dir[ 'basedir' ], $theme_name ) );
				// Create folder if it doesn't exist yet
				if ( !file_exists( $dir ) ) {
					wp_mkdir_p( $dir );
				}
			} else {
				$dir = apply_filters( 'forqy_less_cache_url', path_join( $upload_dir[ 'baseurl' ], $theme_name ) );
			}

			return rtrim( $dir, '/' );
		}

		/**
		 * Adds an Interface to Register 'lessc' Functions. See the Documentation on http://leafo.net/lessphp/docs/#custom_functions
		 *
		 * @param $name
		 * @param $callable
		 */
		public function register( $name, $callable ) {
			$this->registered_functions[ $name ] = $callable;
		}

		/**
		 * Unregister Functions
		 *
		 * @param $name
		 */
		public function unregister( $name ) {
			$this->unregistered_functions[ $name ] = $name;
		}

		/**
		 * Add LESS Var Prior to Compiling
		 *
		 * @param $name
		 * @param $value
		 */
		public function add_var( $name, $value ) {
			if ( is_string( $name ) ) {
				$this->vars[ $name ] = $value;
			}
		}

		/**
		 * Removes a LESS Var
		 *
		 * @param $name
		 */
		public function remove_var( $name ) {
			if ( isset( $this->vars[ $name ] ) ) {
				unset( $this->vars[ $name ] );
			}
		}
	}

	if ( !function_exists( 'register_less_function' ) && !function_exists( 'unregister_less_function' ) ) {

		/**
		 * Register Additional Functions You Can Use in Your LESS Stylesheets
		 *
		 * @param $name
		 * @param $callable
		 */
		function register_less_function( $name, $callable ) {
			$less = forqy_less::instance();
			$less->register( $name, $callable );
		}

		/**
		 * Remove Any Registered 'lessc' Functions
		 *
		 * @param $name
		 */
		function unregister_less_function( $name ) {
			$less = forqy_less::instance();
			$less->unregister( $name );
		}
	}

	if ( !function_exists( 'add_less_var' ) && !function_exists( 'remove_less_var' ) ) {

		/**
		 * A Simple Method of Adding LESS Vars Via a Function Call
		 *
		 * @param $name
		 * @param $value
		 */
		function add_less_var( $name, $value ) {
			$less = forqy_less::instance();
			$less->add_var( $name, $value );
		}

		/**
		 * Remove LESS Vars by an Array Key
		 *
		 * @param $name
		 */
		function remove_less_var( $name ) {
			$less = forqy_less::instance();
			$less->remove_var( $name );
		}

	}

}
